<?php

/*
 * Plugin Name: WooCommerce Conto Quickpay Payment Gateway
 * Plugin URI: https://www.contomobile.com/
 * Description: Allows e-shops to receive their payments online from buyers, who have a bank account or payment card, to pay for goods or services.
 * Author: Contomobile UAB
 * Author URI: https://www.contomobile.com/
 * Version: 2.1.6.3
 * Requires at least: 5.3
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: woocommerce-conto-quickpay
 * Domain Path: /languages
 */

register_activation_hook(__FILE__, function () {
    check_environment();
});

function check_environment()
{

    if (version_compare(phpversion(), '5.4.0', '<')) {
        $message = __(' The minimum PHP version required for Conto Quickpay Payment Gateway is %1$s. You are running %2$s.', 'woocommerce-conto-quickpay');

        die(sprintf($message, '5.4.0', phpversion()));
    }

    if (! defined('WC_VERSION')) {
        die(__('WooCommerce needs to be activated.', 'woocommerce-conto-quickpay'));
    }

    if (version_compare(WC_VERSION, '3.0.0', '<')) {
        $message = __('The minimum WooCommerce version required for Conto Quickpay Payment Gateway is %1$s. You are running %2$s.', 'woocommerce-conto-quickpay');

        die(sprintf($message, '2.0.0', WC_VERSION));
    }

    return true;
}

/**
 * Init plugin.
 */
function woocommerce_conto_quickpay_init()
{
    require_once plugin_basename('library/php-jwt/src/JWT.php');
    require_once plugin_basename('includes/class-wc-jwt-payload.php');
    require_once plugin_basename('includes/class-wc-jwt-payload-card.php');
    require_once plugin_basename('includes/class-wc-jwt-util.php');
    require_once plugin_basename('includes/class-wc-callback-data.php');
    require_once plugin_basename('includes/class-wc-gateway-conto-quickpay-trait.php');
    require_once plugin_basename('includes/class-wc-gateway-conto-quickpay-credit-card.php');
    require_once plugin_basename('includes/class-wc-gateway-conto-quickpay.php');
    require_once plugin_basename('includes/class-wc-bank-selector.php');
    add_filter('woocommerce_payment_gateways', 'contomobile_add_gateway_class');

    // Registers WooCommerce Blocks integration.
    add_action('woocommerce_blocks_loaded', 'woocommerce_gateway_conto_woocommerce_block_support');
    add_action('woocommerce_blocks_loaded', 'woocommerce_gateway_conto_woocommerce_credit_card_block_support');
}

add_action('plugins_loaded', 'woocommerce_conto_quickpay_init', 0);

/**
 * Add the gateway to WooCommerce
 */
function contomobile_add_gateway_class($gateways)
{
    $gateways[] = 'WC_Gateway_Conto_Quickpay';
    $gateways[] = 'WC_Gateway_Conto_Quickpay_Credit_Card';

    return $gateways;
}

add_action('wp_ajax_nopriv_get_banks', 'contomobile_ajax_get_banks');
add_action('wp_ajax_get_banks', 'contomobile_ajax_get_banks');
add_action('wp_ajax_nopriv_get_other_banks', 'contomobile_ajax_get_other_banks');
add_action('wp_ajax_nopriv_get_description', 'contomobile_ajax_get_description');
add_action('wp_ajax_get_description', 'contomobile_ajax_get_description');
add_action('woocommerce_store_api_checkout_update_order_meta', 'conto_quickpay_store_order_meta', 10, 1);

function contomobile_ajax_get_banks()
{
    wp_send_json_success([
        'data' => BankSelector::load_bank_selection_list(),
    ]);
}

function contomobile_ajax_get_description()
{
    wp_send_json_success([
        'data' => BankSelector::get_description(),
    ]);
}

add_action('wp_ajax_nopriv_get_token', 'contomobile_ajax_get_token');
add_action('wp_ajax_get_token', 'contomobile_ajax_get_token');

function contomobile_ajax_get_token()
{
    if (empty($_POST['product_key'])) {
        return;
    }
    $order_id = wc_get_order_id_by_order_key($_POST['product_key']);
    WC()->session->set('cq_order_id', $order_id);
    if (isset($_POST['gateway']) && $_POST['gateway'] == 'quickpay') {
        $gateway = new WC_Gateway_Conto_Quickpay();
        $host    = $gateway->host ? trim($gateway->host, ' \/') : 'https://quickpay.contomobile.com';
    } else {
        $gateway = new WC_Gateway_Conto_Quickpay_Credit_Card();
        $host    = $gateway->host ? trim($gateway->host, ' \/') : 'https://dev.forbis.lt';
    }
    $token = $gateway->get_payload_token($order_id);

    wp_send_json_success([
        'token' => $token,
        'host' => $host,
        'redirectUrlSuccess' => get_home_url() . '/wc-api/cq_success',
        'redirectUrlCancel' => get_home_url() . '/wc-api/cq_cancel',
    ]);
}


add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'plugin_action_links');
function plugin_action_links($links)
{
    $plugin_links = [
        '<a href="' . admin_url( 'admin.php?page=wc-settings&tab=checkout&section=woocommerce-conto-quickpay' ) . '">' . __( 'Settings', 'woocommerce-conto-quickpay' ) . '</a>',
    ];

    return array_merge($plugin_links, $links);
}

function conto_quickpay_store_order_meta($order)
{
    if (!($order instanceof \WC_Order)) {
        return;
    }

    $request_data = json_decode(file_get_contents('php://input'), true);

    if (is_array($request_data)) {
        $saved = false;
        if (isset($request_data['enable_periodic_payment'])) {
            $enable_periodic = sanitize_text_field($request_data['enable_periodic_payment']);
            $order->update_meta_data('_enable_periodic_payment', $enable_periodic);
            $saved = true;
        }

        if (isset($request_data['start_date'])) {
            $start_date = sanitize_text_field($request_data['start_date']);
            $order->update_meta_data('_start_date', $start_date);
            $saved = true;
        }

        if (isset($request_data['end_date'])) {
            $end_date = sanitize_text_field($request_data['end_date']);
            $order->update_meta_data('_end_date', $end_date);
            $saved = true;
        }

        if (isset($request_data['payment_frequency'])) {
            $frequency = sanitize_text_field($request_data['payment_frequency']);
            $order->update_meta_data('_payment_frequency', $frequency);
            $saved = true;
        }

        if ($saved) {
            $order->save();
        }
    }
}

/**
 * Load plugin textdomain.
 */
function woocommerce_conto_quickpay_load_textdomain()
{
    load_plugin_textdomain('woocommerce-conto-quickpay', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

add_action('plugins_loaded', 'woocommerce_conto_quickpay_load_textdomain');



function woocommerce_gateway_conto_woocommerce_block_support()
{
    if (class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        require_once 'includes/blocks/class-wc-gateway-conto-quickpay-blocks.php';
        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function (Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
                $payment_method_registry->register(new WC_Gateway_Conto_Blocks_Support());
            }
        );
    }
}

function woocommerce_gateway_conto_woocommerce_credit_card_block_support()
{
    if (class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        require_once 'includes/blocks/class-wc-gateway-conto-quickpay-credit-card-blocks.php';
        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function (Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
                $payment_method_registry->register(new WC_Gateway_Conto_Credit_Card_Blocks_Support());
            }
        );
    }
}
