<?php

/**
 * Provides a Conto Quickpay Payment Gateway.
 */
class WC_Gateway_Conto_Quickpay extends WC_Payment_Gateway
{
    use WC_Gateway_Conto_Quickpay_Trait;

    public const CONTO_HOST = 'https://quickpay.contomobile.com';
    public const CONTO_WIDGET_JS = 'https://quickpay.contomobile.com/static/widget.js';

    /**
     * Contract ID given by QuickPay provider.
     *
     * @var integer
     */
    protected $contract_id;

    /**
     * Secret used for generating/verifying signatures in tokens between
     * QuickPay services. This value is given by QuickPay provider.
     *
     * @var string
     */
    protected $secret_key;

    /**
     * How load QuickPay(in popup or redirect).
     *
     * @var string
     */
    public $popup = 'no';

    /**
     * Whether or not using other banks.
     *
     * @var bool
     */
    public $other_banks;

    /**
     * Whether or not using periodic payments.
     *
     * @var bool
     */
    public $periodic_payments;

    /**
     * Periodic payment remittance information.
     *
     * @var bool
     */
    public $remittance_information;

    /**
     * Whether or not logging is enabled
     *
     * @var bool
     */
    public static $log_enabled = true;

    /**
     * Logger instance
     *
     * @var WC_Logger
     */
    public static $log;

    /**
     * Whether or not use WC blocks.
     *
     * @var bool
     */
    public $use_wc_blocks;

    /**
     * Constructor
     */
    public function __construct()
    {

        if (version_compare(phpversion(), '7.1', '>=')) {
            ini_set('serialize_precision', '-1');
        }

        $this->id                 = 'woocommerce-conto-quickpay';
        $this->icon               = WP_PLUGIN_URL . '/' . plugin_basename(dirname(__DIR__)) . '/assets/img/logo.png';
        $this->has_fields         = true;
        $this->method_title       = __('Conto QuickPay', 'woocommerce-conto-quickpay');
        $this->method_description = __('Allows e-shops to receive their payments online from buyers, who have a bank account but do not want to use a payment card.', 'woocommerce-conto-quickpay');

        $this->supports = array(
            'products',
            'subscriptions',
            'subscription_cancellation',
            'subscription_suspension',
            'subscription_reactivation',
            'subscription_amount_changes',
            'subscription_date_changes',
            'multiple_subscriptions',
        );

        $this->init_form_fields();
        $this->init_settings();
        $currency = get_woocommerce_currency();

        if ($currency != 'EUR' && $this->get_option('enabled') == 'yes') {
            $this->update_option('enabled', 'no');
            $this->enabled = 'no';
            add_action('admin_notices', $this->add_admin_notice('Conto QuickPay. You can\'t enable this gateway because your currency is not Euro. Please, change this in Woocommerce settings and try again.'));

            return false;
        } else {
            $this->enabled = $this->get_option('enabled');
        }

        if (empty($this->get_option('title'))) {
            $this->title = __('Conto QuickPay', 'woocommerce-conto-quickpay');
        } else {
            $this->title = $this->get_option('title');
        }

        $this->description = $this->get_option('description');
        $this->popup       = $this->get_option('popup');
        $this->signed      = $this->get_option('signed');
        $this->confirmed   = $this->get_option('confirmed');
        $this->failed      = $this->get_option('failed');
        $this->expired     = $this->get_option('expired');
        $this->active      = $this->get_option('active');
        $this->cancelled   = $this->get_option('cancelled');
        $this->unresolved  = $this->get_option('unresolved');
        $this->contract_id = $this->get_option('contract_id');
        $this->secret_key  = $this->get_option('secret_key');
        $this->host        = $this->get_option('host');

        $this->email_shop_name       = $this->get_option('email_shop_name');
        $this->email_shop_url        = $this->get_option('email_shop_url');
        $this->email_send_on_success = $this->get_option('email_send_on_success');
        $this->email_send_on_failure = $this->get_option('email_send_on_failure');
        $this->use_wc_blocks         = $this->get_option('use_wc_blocks');

        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [
            $this,
            'process_admin_options',
        ]);

        add_action('wp_enqueue_scripts', [$this, 'payment_scripts']);

        add_action('woocommerce_api_cq_success', [
            $this,
            'webhook_cq_success',
        ]);
        add_action('woocommerce_api_cq_cancel', [
            $this,
            'webhook_cq_cancel',
        ]);
        add_action('woocommerce_api_conto-quickpay/callback', [
            $this,
            'webhook_callback',
        ]);

        wp_enqueue_script('conto-quickpay-multiselect-js', plugin_dir_url(__DIR__) . 'assets/admin/conto-quickpay-multiselect.js', ['jquery'], 0.1, false);
    }

    /**
     * Initialise Gateway Settings Form Fields
     */
    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled' => [
                'title' => __('Enable/Disable', 'woocommerce-conto-quickpay'),
                'label' => __('Enable Contomobile Gateway', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => '',
                'default' => 'no',
            ],
            'title' => [
                'title' => __('Title', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'woocommerce-conto-quickpay'),
                'default' => __('Conto QuickPay', 'woocommerce-conto-quickpay'),
            ],
            'description' => [
                'title' => __('Description', 'woocommerce-conto-quickpay'),
                'type' => 'textarea',
                'description' => __('This controls the description which the user sees during checkout.', 'woocommerce-conto-quickpay'),
                'default' => __('Allows e-shops to receive their payments online from buyers, who have a bank account but do not want to use a payment card.', 'woocommerce-conto-quickpay'),
            ],
            'popup' => [
                'title' => 'Pop-Up',
                'label' => __('Open payment in popup?', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Open payment in popup without redirect to Conto QuickPay site.', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'signed' => [
                'title' => __('Signed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Signed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'confirmed' => [
                'title' => __('Confirmed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Confirmed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'failed' => [
                'title' => __('Failed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Failed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'expired' => [
                'title' => __('Expired', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Expired payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'active' => [
                'title' => __('Active', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Active periodic/recurring payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'cancelled' => [
                'title' => __('Cancelled', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Cancelled periodic/recurring payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'unresolved' => [
                'title' => __('Unresolved', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Unresolved periodic/recurring payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'contract_id' => [
                'title' => __('Contract ID', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Provided by Conto QuickPay.', 'woocommerce-conto-quickpay'),
            ],
            'secret_key' => [
                'title' => __('Secret key', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Provided by Conto QuickPay.', 'woocommerce-conto-quickpay'),
            ],
            'host' => [
                'title' => __('Conto QuickPay widget host', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Add host URL e.g. https://quickpay.contomobile.com. If you don\'t have this value, please leave it empty.', 'woocommerce-conto-quickpay'),
            ],
            'bank_selection' => [
                'title' => __('Bank Selection', 'woocommerce-conto-quickpay'),
                'type' => 'multiselect',
                'class' => 'conto-quickpay-multiselect',
                'css' => 'width: 400px;',
                'default' => '',
                'description' => __('Select banks that you want to activate (keep in mind that not all banks might be active).', 'woocommerce-conto-quickpay'),
                'options' => BankSelector::load_bank_selection_list_for_admin(),
                'desc_tip' => true,
                'custom_attributes' => [
                    'data-placeholder' => __('Select shipping methods', 'woocommerce'),
                ],
            ],
            'other_banks' => [
                'title' => __('Other banks', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Select/Unselect all other country financial institutions', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'periodic_payments' => [
                'title' => __('Turn on periodic payments', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Enable/Disable periodic payments (enable only if allowed on FORPOST contract)'),
                'default' => 'no',
                'options' => [
                    'yes' => __('Yes', 'woocommerce-conto-quickpay'),
                    'no' => __('No', 'woocommerce-conto-quickpay'),
                ],
            ],
            'remittance_information' => [
                'title' => __('Remittance information of periodic payment', 'woocommerce-conto-quickpay'),
                'type' => 'textarea',
                'description' => __('Specifying a purpose when creating a periodic payment agreement is optional.', 'woocommerce-conto-quickpay'),
                'custom_attributes' => [
                    'maxlength' => 140,
                ],
            ],
            'email_shop_name' => [
                'title' => __('Shop Name', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Shop name that will be displayed in the payer\'s e-mail notification. Example: Your Shop', 'woocommerce-conto-quickpay'),
            ],
            'email_shop_url' => [
                'title' => __('Shop URL', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Shop URL that will be displayed in payer\'s e-mail notification. Example: your-shop.com', 'woocommerce-conto-quickpay'),
            ],
            'email_send_on_success' => [
                'title' => __('Send if succeeded', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Send e-mail to payer if payment has been successful', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'email_send_on_failure' => [
                'title' => __('Send if failed', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Send e-mail to payer if payment failed', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'use_wc_blocks' => [
                'title' => __('WooCommerce Blocks', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Need to activate if use WooCommerce Blocks for checkout', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
        ];
    }

    public function validate_remittance_information_field($key, $value)
    {
        $value = sanitize_textarea_field($value);

        if (strlen($value) > 140) {
            WC_Admin_Settings::add_error(__('Remittance information must not exceed 140 characters.', 'woocommerce-conto-quickpay'));
            return substr($value, 0, 140);
        }

        return $value;
    }

    public function generate_multiselect_html($key, $data)
    {
        $data['select_unselect_all'] = __('Select/Unselect all', 'woocommerce-conto-quickpay');
        $field_key = $this->get_field_key($key);
        $defaults = [
            'title' => '',
            'disabled' => false,
            'class' => '',
            'css' => '',
            'placeholder' => '',
            'type' => 'text',
            'desc_tip' => false,
            'description' => '',
            'custom_attributes' => [],
            'select_buttons' => false,
            'options' => [],
        ];

        $data = wp_parse_args($data, $defaults);
        $banksSettings = $this->get_option('bank_selection');

        ob_start();
        ?>
        <tr valign="top">
            <th scope="row" class="titledesc">
                <label
                        for="<?php echo esc_attr($field_key); ?>"><?php echo wp_kses_post($data['title']); ?><?php echo $this->get_tooltip_html($data); // WPCS: XSS ok.
        ?></label>
            </th>
            <td class="forminp">
                <label><input type="checkbox" name="select-all"
                              id="select-all"/><b><?php echo $data['select_unselect_all']; ?></b></label><br/>
                <legend class="screen-reader-text">
                    <span><?php echo wp_kses_post($data['title']); ?></span></legend>
                <style type="text/css">
                    .conto-quickpay-bank-option-container {
                        position: relative;
                        float: left;
                        width: 150px;
                        height: 60px;
                        margin-right: 30px;
                    }
                </style>
                <?php foreach ((array)$data['options'] as $option_key => $option_value) : ?>
                    <div class="conto-quickpay-bank-option-container">
                        <label>
                            <div
                                    style="margin: 0;position: absolute;top: 50%;-ms-transform: translateY(-50%);transform: translateY(-50%);">
                                <input
                                        type="checkbox" <?php if (is_array($banksSettings) && array_key_exists($option_key, $banksSettings)) {
                                            echo 'checked';
                                        } ?>
                                        name="woocommerce_woocommerce-conto-quickpay_bank_selection[<?php echo $option_key; ?>]"
                                        id="woocommerce_woocommerce-conto-quickpay_bank_selection[<?php echo $option_key; ?>]">
                            </div>
                            <div
                                    style="margin: 0;position: absolute;top: 50%;-ms-transform: translateY(-50%);transform: translateY(-50%);left: 20%;">
                                <?php echo $option_value; ?>
                            </div>
                        </label>
                    </div>
                <?php endforeach; ?>
            </td>
        </tr>
        <?php

        return ob_get_clean();
    }

    /**
     * Custom CSS and JS, in most cases required only when you decided to go
     * with a custom credit card form.
     */
    public function payment_scripts()
    {
        if (!is_cart() && !is_checkout() && !isset($_GET['pay_for_order'])) {
            return;
        }

        if ('no' === $this->enabled) {
            return;
        }
        wp_enqueue_script('conto_js', self::CONTO_WIDGET_JS, [
            'jquery',
        ]);

        wp_enqueue_script('wp-util');

        wp_register_script('woocommerce_conto', WP_PLUGIN_URL . '/' . plugin_basename(dirname(dirname(__FILE__))) . '/assets/js/conto.js', [
            'jquery',
            'conto_js',
            'wp-util',
        ]);

        wp_enqueue_script('woocommerce_conto');

        wp_enqueue_style('custom-frontend-style', WP_PLUGIN_URL . '/' . plugin_basename(dirname(__DIR__)) . '/assets/css/quickpay.css');

        wp_add_inline_script('woocommerce_conto', 'is_popup = ' . json_encode(array(
                'value' => $this->popup,
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'is_banklist = ' . json_encode(array(
                'value' => $this->get_option('bank_selection'),
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'is_other_banks = ' . json_encode(array(
                'value' => $this->get_option('other_banks'),
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'is_periodic_payments = ' . json_encode(array(
                'value' => $this->get_option('periodic_payments'),
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'remittance_information = ' . json_encode(array(
                'value' => $this->get_option('remittance_information'),
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'wc_nonce = ' . json_encode(array(
                'value' => wp_create_nonce('wc_store_api'),
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'optional = ' . json_encode(array(
                'value' => '(' . __('Optional', 'woocommerce-conto-quickpay') . ')',
            )), 'before');
        wp_add_inline_script('woocommerce_conto', 'required = ' . json_encode(array(
                'value' => '(' . __('Required', 'woocommerce-conto-quickpay') . ')',
            )), 'before');
    }

    public function process_payment($order_id)
    {
        $selected_bank = isset($_POST['bank']) ? $_POST['bank'] : null;
        $selected_banks = $this->get_option('bank_selection');
        $creditors = !empty($selected_banks)
            ? implode(',', array_values(array_keys($this->get_option('bank_selection'))))
            : null;

        $creditor = $_POST['bank'] ?? $creditors;
        $otherBanks = $this->other_banks == 'yes' ? true : false;
        $periodicPayments = $this->periodic_payments == 'yes' ? true : false;
        $token = $this->get_payload_token($order_id);
        $host = $this->host ? trim($this->host, ' \/') : self::CONTO_HOST;
        WC()->session->set('cq_order_id', $order_id);

        $data = [
            'token' => $token,
            'redirectUrlSuccess' => get_home_url() . '/wc-api/cq_success',
            'redirectUrlCancel' => get_home_url() . '/wc-api/cq_cancel',
            'creditor' => $creditor,
            'other' => $otherBanks,
            'webview' => false,
        ];

        if ($periodicPayments && isset($_POST['enable_periodic_payment']) && $_POST['enable_periodic_payment'] === 'yes') {
            $data['periodicPayment'] = [
                'interval' => $_POST['payment_frequency'] ?? null,
                'startDate' => $_POST['start_date'] ?? null,
                'endDate' => $_POST['end_date'] ?? null,
                'remittanceInformation' => $_POST['remittance_information'] ?? $this->get_option('remittance_information'),
            ];
        }

        if ($this->popup == 'yes') {
            $data['host'] = $host;
            $data['webview'] = false;

            wp_send_json($data);
        }

        if ($this->use_wc_blocks == 'yes') {
            $data['host'] = $host;
            wp_send_json($data);
        }

        $redirect = $host . '/initialize?' . http_build_query($data);

        return [
            'result' => 'success',
            'redirect' => $redirect,
        ];
    }

    /**
     * Conto Quickpay success Payment process.
     */
    public function webhook_cq_success()
    {
        global $woocommerce;
        $order_id = WC()->session->get('cq_order_id');
        $this->log('Success redirect page: session cq_order_id = ' . $order_id);
        $order_status = 'wc-pending';
        $order        = wc_get_order($order_id);
        $url          = $this->get_return_url($order);

        if (!empty($order)) {
            $current_status = $order->get_status();

            $confirmed_status = str_replace(['wc-', ' '], ['', '-'], $this->get_option('confirmed'));
            $active_status = str_replace(['wc-', ' '], ['', '-'], $this->get_option('active'));

            if (!in_array($current_status, ['wc-processing', 'wc-completed', 'processing', 'completed', $confirmed_status, $active_status])) {
                $order->set_status($order_status);
            }

            $order->save();
            WC()->session->__unset('cq_order_id');

            if (isset(WC()->cart)) {
                WC()->cart->empty_cart();
            }

            $this->log('Order id: ' . $order_id . ' was successfully placed.');
            wp_redirect($url);
            exit();
        } else {
            wp_redirect(home_url());
            $this->log('Can\'t get an Order, make sure the transaction ID is correct.', 'critical');
        }
    }

    /**
     * Conto Quickpay cancel Payment process.
     */
    public function webhook_cq_cancel()
    {
        $order_id         = WC()->session->get('cq_order_id');
        $order_status     = 'wc-cancelled';
        $order            = wc_get_order($order_id);
        $confirmed_status = str_replace(['wc-', ' '], ['', '-'], $this->get_option('confirmed'));
        $active_status = str_replace(['wc-', ' '], ['', '-'], $this->get_option('active'));

        if (!empty($order)) {
            $current_status = $order->get_status();

            if (!in_array($current_status, ['wc-processing', 'wc-completed', 'processing', 'completed', $confirmed_status, $active_status])) {
                $order->set_status($order_status);
            }

            $order->save();
            WC()->session->__unset('cq_order_id');
            $this->log('Order id: ' . $order_id . ' was canceled.');
            wp_redirect(wc_get_checkout_url());
            exit();
        } else {
            wp_redirect(home_url());
            $this->log('Can\'t get an Order, make sure the transaction ID is correct.', 'critical');
        }
    }

    /**
     * Conto Quickpay callback Payment process.
     */
    public function webhook_callback()
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $callbackData = $this->getCallbackData();
            $transactions = $callbackData->getTransactios();

            if (!empty($transactions)) {
                $wrong_transaction = [];
                foreach ($transactions as $transaction) {
                    $transactionId = $transaction->transactionId;

                    if (!in_array($transaction->status, [
                        CallbackData::STATUS_CONFIRMED,
                        CallbackData::STATUS_SIGNED,
                        CallbackData::STATUS_EXPIRED,
                        CallbackData::STATUS_FAILED,
                        CallbackData::STATUS_ACTIVE,
                        CallbackData::STATUS_CANCELLED,
                        CallbackData::STATUS_UNRESOLVED,
                    ])) {
                        $wrong_transaction[] = $transactionId;
                        continue;
                    }

                    $order_id = $this->decrypt($transactionId);

                    if (!$order_id) {
                        $wrong_transaction[] = $transactionId;
                        continue;
                    }

                    $status_transaction = wc_strtolower($transaction->status);

                    switch ($status_transaction) {
                        case 'confirmed':
                            $tr_status = $this->get_option('confirmed');
                            break;
                        case 'signed':
                            $tr_status = $this->get_option('signed');
                            break;
                        case 'expired':
                            $tr_status = $this->get_option('expired');
                            break;
                        case 'failed':
                            $tr_status = $this->get_option('failed');
                            break;
                        case 'active':
                            $tr_status = $this->get_option('active');
                            break;
                        case 'cancelled':
                            $tr_status = $this->get_option('cancelled');
                            break;
                        case 'unresolved':
                            $tr_status = $this->get_option('unresolved');
                            break;
                        default:
                            $wrong_transaction[] = $transactionId;
                            break;
                    }

                    $order = wc_get_order($order_id);
                    if (!empty($order)) {
                        $callback_status = str_replace(['wc-', ' '], [
                            '',
                            '-',
                        ], $tr_status);

                        $current_status = $order->get_status();

                        if ($current_status == str_replace(['wc-', ' '], ['', '-',], $this->get_option('confirmed'))
                            && $callback_status == str_replace(['wc-', ' '], ['', '-',], $this->get_option('failed'))) {
                            $wrong_transaction[] = $transactionId;
                            $this->log('Can\'t change the status from confirmed to failed. Transaction [' . $transactionId . ']', 'critical');
                            continue;
                        }

                        $order->set_status($callback_status);
                        $order->save();
                        $this->log('Callback Payment process: Order id = ' . $order_id . '. Order key = ' . $transactionId . ' Transaction status = ' . $status_transaction);
                    } else {
                        $wrong_transaction[] = $transactionId;
                        $this->log('Wrong transactions[' . implode(', ', $wrong_transaction) . ']', 'critical');
                    }
                }
                if (!empty($wrong_transaction)) {
                    $this->sendFailResponse('Wrong transactions: [' . implode(', ', $wrong_transaction) . ']');
                } else {
                    $this->sendOkResponse();
                }
            } else {
                $this->sendFailResponse('Transactions are empty.');
            }
        } else {
            $this->log('Request method \'' . $_SERVER['REQUEST_METHOD'] . '\' not allowed. Only POST can be used.', 'critical');
            $this->sendFailResponse('Request method \'' . $_SERVER['REQUEST_METHOD'] . '\' not allowed. Only POST can be used.', 404);
            wp_redirect(home_url());
            exit();
        }
    }
}
