<?php

trait WC_Gateway_Conto_Quickpay_Trait
{
    /**
     * Custom notice for admin.
     */
    public function add_admin_notice($message)
    {
        ?>
        <div class="error"><p><?php echo $message; ?></p></div>
        <?php
    }

    private function encrypt($data)
    {
        $cipherAlgorithm = 'aes-256-ctr';

        $secretKey  = $this->secret_key;
        $passphrase = openssl_digest($secretKey, 'sha256');

        $ivSize = openssl_cipher_iv_length($cipherAlgorithm);
        $iv     = openssl_random_pseudo_bytes($ivSize);

        $cipherOptions = OPENSSL_RAW_DATA;

        $cipherText = openssl_encrypt($data, $cipherAlgorithm, $passphrase, $cipherOptions, $iv);

        $b64EncodedCipherText = base64_encode($iv . $cipherText);

        return $b64EncodedCipherText;
    }

    private function decrypt($b64EncodedCipherText)
    {
        $cipherAlgorithm = 'aes-256-ctr';

        $secretKey  = $this->secret_key;
        $passphrase = openssl_digest($secretKey, 'sha256');

        $ivSize = openssl_cipher_iv_length($cipherAlgorithm);

        $decoded = base64_decode($b64EncodedCipherText);
        if ($decoded === false) {
            self::log("Base64 decode failed for encrypted data", 'error');
            return false;
        }

        if (strlen($decoded) < $ivSize) {
            self::log("Invalid encrypted data: too short", 'error');
            return false;
        }
        $iv = substr($decoded, 0, $ivSize);
        $cipherText = substr($decoded, $ivSize);

        $cipherOptions = OPENSSL_RAW_DATA;

        $plainText = openssl_decrypt($cipherText, $cipherAlgorithm, $passphrase, $cipherOptions, $iv);
        if ($plainText === false) {
            self::log("Decryption failed: invalid ciphertext or key", 'error');
            return false;
        }

        return $plainText;
    }

    /**
     * Get payload token that use for Conto Quickpay request.
     */
    public function get_payload_token($order_id)
    {
        $order = wc_get_order($order_id);
        if (!$order) {
            self::log("Order not found for ID: {$order_id}", 'error');
            return false;
        }

        $amount      = $order->get_total();
        $contract_id = $this->contract_id;
        $secret      = $this->secret_key;
        $payment_purpose = 'Order payment #' . $order_id;

        if ($order->get_meta('_enable_periodic_payment') === 'yes') {
            $enable_periodic = $order->get_meta('_enable_periodic_payment') ?? null;
            $start_date      = $order->get_meta('_start_date') ?? null;
            $end_date        = $order->get_meta('_end_date') ?? null;
            $frequency       = $order->get_meta('_payment_frequency') ?? null;
        } elseif (sanitize_text_field($_POST['enable_periodic_payment']) === 'yes') {
            $enable_periodic = sanitize_text_field($_POST['enable_periodic_payment']) ?? null;
            $start_date      = sanitize_text_field($_POST['start_date']) ?? null;
            $end_date        = sanitize_text_field($_POST['end_date']) ?? null;
            $frequency       = sanitize_text_field($_POST['payment_frequency']) ?? null;
        }

        if ($order->get_meta('_enable_periodic_payment') === 'yes' || sanitize_text_field($_POST['enable_periodic_payment']) === 'yes') {
            $remittance_information = get_option('woocommerce_woocommerce-conto-quickpay_settings')['remittance_information'] ?: $payment_purpose;
        } else {
            $remittance_information = $payment_purpose;
        }

        $jwtPayload = JWTPayload::factory()
            ->setAmount($amount)
            ->setContractId($contract_id)
            ->setPaymentPurpose($remittance_information)
            ->setTransactionId($this->encrypt($order_id))
            ->setCheckoutId($order_id)
            ->setSendOnSuccess($this->email_send_on_success === 'yes')
            ->setSendOnFailure($this->email_send_on_failure === 'yes')
            ->setShopName($this->email_shop_name)
            ->setShopUrl($this->email_shop_url)
            ->setAddress($this->encrypt($order->get_billing_email()));

        if (isset($enable_periodic) && $enable_periodic === 'yes') {
            $jwtPayload->setEnablePeriodicPayment(true)
                ->setServiceType('PERIODIC')
                ->setPeriodicStartDate($start_date)
                ->setPeriodicFrequency($frequency)
                ->setPeriodicEndDate($end_date);
        }

        $payload = $jwtPayload->create();

        return JWTUtil::encode($payload, $secret, $algorithm = JWTUtil::defaultAlg);
    }

    public function get_payload_token_card($order_id)
    {
        $order       = wc_get_order($order_id);
        $amount      = $order->get_total();
        $contract_id = $this->contract_id;
        $secret      = $this->secret_key;

        $jwtPayload = JWTPayloadCard::factory()
            ->setAmount($amount)
            ->setContractId($contract_id)
            ->setPaymentPurpose('Order payment #' . $order_id)
            ->setTransactionId($this->encrypt($order_id))
            ->setCheckoutId($order_id)
            ->setSendOnSuccess($this->email_send_on_success === 'yes')
            ->setSendOnFailure($this->email_send_on_failure === 'yes')
            ->setShopName($this->email_shop_name)
            ->setShopUrl($this->email_shop_url)
            ->setAddress($this->encrypt($order->get_billing_email()))
            ->create();

        return JWTUtil::encode($jwtPayload, $secret, $algorithm = JWTUtil::defaultAlg);
    }

    /**
     * Provides Conto Quickpay CallbackData.
     */
    private function getCallbackData()
    {
        $jwtSecret  = $this->secret_key;
        $jwtSignAlg = JWTUtil::defaultAlg;
        try {
            $jwtToken = $this->parseRequestBody();

            return new CallbackData(JWTUtil::decode($jwtToken, $jwtSecret, $jwtSignAlg));
        } catch (Exception $e) {
            $this->sendFailResponse('JWT Decode Error: \'' . $e->getMessage() . '\'');
        }
        exit();
    }

    /**
     * Parse callback request from Conto Quickpay.
     */
    private function parseRequestBody()
    {
        try {
            set_error_handler(function ($severity, $message) {
                throw new Exception($message, $severity);
            });
            $body     = file_get_contents('php://input');
            $json     = json_decode($body, true);
            $jwtToken = $json['token'];
            restore_error_handler();

            return $jwtToken;
        } catch (Exception $e) {
            $this->sendFailResponse('POST Request Body Parse Error: \'' . $e->getMessage() . '\'');
            exit();
        }
    }

    /**
     * Send OK status.
     */
    private function sendOkResponse()
    {
        wp_send_json([ 'status' => 'ok' ], 200);
    }

    /**
     * Send error status.
     */
    private function sendFailResponse($message, $status_code = 400)
    {
        wp_send_json(array(
            'status'  => 'failure',
            'message' => $message,
        ), $status_code);
    }

    /**
     * Logging method.
     *
     * @param string $message Log message.
     * @param string $level Optional. Default 'info'. Possible values:
     *                      emergency|alert|critical|error|warning|notice|info|debug.
     */
    public static function log($message, $level = 'info')
    {
        if (self::$log_enabled) {
            if (empty(self::$log)) {
                self::$log = wc_get_logger();
            }
            self::$log->log($level, $message, [ 'source' => 'woocommerce-conto-quickpay' ]);
        }
    }
}
