<?php

/**
 * Provides a Conto Quickpay Payment Gateway by Credit card.
 */
class WC_Gateway_Conto_Quickpay_Credit_Card extends WC_Payment_Gateway
{
    use WC_Gateway_Conto_Quickpay_Trait;

    public const CONTO_CARD_HOST      = 'https://cardpayments.contomobile.com';
    public const CONTO_CARD_WIDGET_JS = 'https://cardpayments.contomobile.com/static/card-widget.js';

    /**
     * Contract ID given by QuickPay provider.
     *
     * @var integer
     */
    protected $contract_id;

    /**
     * Secret used for generating/verifying signatures in tokens between
     * QuickPay services. This value is given by QuickPay provider.
     *
     * @var string
     */
    protected $secret_key;

    /**
     * How load QuickPay(redirect or iframe).
     *
     * @var string
     */
    public $embedded;

    /**
     * Whether or not use WC blocks.
     *
     * @var bool
     */
    public $use_wc_blocks;

    /**
     * Whether or not logging is enabled
     *
     * @var bool
     */
    public static $log_enabled = true;

    /**
     * Logger instance
     *
     * @var WC_Logger
     */
    public static $log;

    /**
     * Constructor
     */
    public function __construct()
    {
        if (version_compare(phpversion(), '7.1', '>=')) {
            ini_set('serialize_precision', '-1');
        }

        $this->id                 = 'woocommerce-conto-quickpay-credit-card';
        $this->icon               = WP_PLUGIN_URL . '/' . plugin_basename(dirname(__DIR__)) . '/assets/img/logo.png';
        $this->has_fields         = true;
        $this->method_title       = __('Conto QuickPay - Card payment', 'woocommerce-conto-quickpay');
        $this->method_description = __('Allows e-shops to receive their payments online from buyers, who have a payment card.', 'woocommerce-conto-quickpay');

        $this->supports = [
            'products',
            'refunds',
            'subscriptions',
        ];

        // Method with all the options fields
        $this->init_form_fields();

        // Load the settings.
        $this->init_settings();
        $currency = get_woocommerce_currency();
        if ($currency != 'EUR' && $this->get_option('enabled') == 'yes') {
            $this->update_option('enabled', 'no');
            $this->enabled = 'no';
            add_action('admin_notices', $this->add_admin_notice('Conto QuickPay. You can\'t enable this gateway because your currency is not Euro. Please, change this in Woocommerce settings and try again.'));

            return false;
        } else {
            $this->enabled = $this->get_option('enabled');
        }
        if (empty($this->get_option('title'))) {
            $this->title = __('Conto Quickpay Card payment', 'woocommerce-conto-quickpay');
        } else {
            $this->title = $this->get_option('title');
        }
        $this->description = $this->get_option('description');
        $this->embedded    = $this->get_option('payment_flow');
        $this->signed      = $this->get_option('signed');
        $this->confirmed   = $this->get_option('confirmed');
        $this->failed      = $this->get_option('failed');
        $this->expired     = $this->get_option('expired');
        $this->contract_id = $this->get_option('contract_id');
        $this->secret_key  = $this->get_option('secret_key');
        $this->host        = rtrim($this->get_option('host'), '/');

        $this->email_shop_name       = $this->get_option('email_shop_name');
        $this->email_shop_url        = $this->get_option('email_shop_url');
        $this->email_send_on_success = $this->get_option('email_send_on_success');
        $this->email_send_on_failure = $this->get_option('email_send_on_failure');
        $this->use_wc_blocks         = $this->get_option('use_wc_blocks');

        // Save settings.
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [
            $this,
            'process_admin_options',
        ]);

        // Add Conto Quickpay payment JS widget.
        add_action('wp_enqueue_scripts', [ $this, 'payment_scripts' ]);

        // Register webhooks.
        add_action('woocommerce_api_cq_card_success', [
            $this,
            'webhook_cq_card_success',
        ]);
        add_action('woocommerce_api_cq_card_cancel', [
            $this,
            'webhook_cq_card_cancel',
        ]);
        add_action('woocommerce_api_conto-quickpay/callback_card', [
            $this,
            'webhook_callback',
        ]);

        wp_enqueue_script('conto-quickpay-multiselect-js', plugin_dir_url(__DIR__) . '/assets/admin/conto-quickpay-multiselect.js', [ 'jquery' ], 0.1, false);
    }


    /**
     * Initialise Gateway Settings Form Fields
     */
    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled' => [
                'title' => __('Enable/Disable', 'woocommerce-conto-quickpay'),
                'label' => __('Enable Contomobile Gateway', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => '',
                'default' => 'no',
            ],
            'title' => [
                'title' => __('Title', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('This controls the title which the user sees during checkout.', 'woocommerce-conto-quickpay'),
                'default' => __('Card payment', 'woocommerce-conto-quickpay'),
            ],
            'payment_flow' => [
                'title' => __('Payment flow', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose embedded or redirect payment flow.', 'woocommerce-conto-quickpay'),
                'options' => [
                    'embedded' => __('Embedded', 'woocommerce-conto-quickpay'),
                    'redirect' => __('Redirect', 'woocommerce-conto-quickpay'),
                ],
            ],
            'description' => [
                'title' => __('Description', 'woocommerce-conto-quickpay'),
                'type' => 'textarea',
                'description' => __('This controls the description which the user sees during checkout.', 'woocommerce-conto-quickpay'),
                'default' => __('Allows e-shops to receive their payments online from buyers, who have a payment card.', 'woocommerce-conto-quickpay'),
            ],
            'signed' => [
                'title' => __('Signed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Signed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'confirmed' => [
                'title' => __('Confirmed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Confirmed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'failed' => [
                'title' => __('Failed', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Failed payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'expired' => [
                'title' => __('Expired', 'woocommerce-conto-quickpay'),
                'type' => 'select',
                'description' => __('Choose order status for Expired payment transaction.', 'woocommerce-conto-quickpay'),
                'options' => wc_get_order_statuses(),
            ],
            'contract_id' => [
                'title' => __('Contract ID', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Provided by Conto QuickPay.', 'woocommerce-conto-quickpay'),
            ],
            'secret_key' => [
                'title' => __('Secret key', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Provided by Conto QuickPay.', 'woocommerce-conto-quickpay'),
            ],
            'host' => [
                'title' => __('Conto QuickPay widget host', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Add host without "/" of the end(e.g. https://dev.forbis.lt/",). If you don\'t have this value, please leave empty.', 'woocommerce-conto-quickpay'),

            ],
            'email_shop_name' => [
                'title' => __('Shop Name', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Shop name that will be displayed in the payer\'s e-mail notification. Example: Your Shop', 'woocommerce-conto-quickpay'),
            ],
            'email_shop_url' => [
                'title' => __('Shop URL', 'woocommerce-conto-quickpay'),
                'type' => 'text',
                'description' => __('Shop URL that will be displayed in payer\'s e-mail notification. Example: your-shop.com', 'woocommerce-conto-quickpay'),
            ],
            'email_send_on_success' => [
                'title' => __('Send if succeeded', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Send e-mail to payer if payment has been successful', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'email_send_on_failure' => [
                'title' => __('Send if failed', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Send e-mail to payer if payment failed', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
            'use_wc_blocks' => [
                'title' => __('WooCommerce Blocks', 'woocommerce-conto-quickpay'),
                'type' => 'checkbox',
                'description' => __('Need to activate if use WooCommerce Blocks for checkout', 'woocommerce-conto-quickpay'),
                'default' => 'no',
            ],
        ];
    }

    /**
     * Custom CSS and JS, in most cases required only when you decided to go
     * with a custom credit card form.
     */
    public function payment_scripts()
    {
        //  Add JS to process a token only on cart/checkout pages.
        if (!is_cart() && !is_checkout() && !isset($_GET['pay_for_order'])) {
            return;
        }

        // If Conto QuickPay payment gateway is disabled, we do not have to enqueue JS.
        if ('no' === $this->enabled) {
            return;
        }
        wp_enqueue_script('conto_card_js', self::CONTO_CARD_WIDGET_JS, [
            'jquery',
        ]);

        wp_enqueue_script('wp-util');

        wp_register_script('woocommerce_card_conto', WP_PLUGIN_URL . '/' . plugin_basename(dirname(__DIR__)) . '/assets/js/contoCard.js', [
            'jquery',
            'conto_card_js',
            'wp-util',
        ]);

        wp_enqueue_script('woocommerce_card_conto');

        wp_add_inline_script('woocommerce_conto', 'is_embedded_card = ' . json_encode([
                'value' => $this->embedded,
            ]), 'before');

        wp_add_inline_script('woocommerce_conto', 'wc_nonce = ' . json_encode([
                'value' => wp_create_nonce('wc_store_api'),
            ]), 'before');

        wp_enqueue_style('custom-frontend-style', WP_PLUGIN_URL . '/' . plugin_basename(dirname(__DIR__)) . '/assets/css/quickpay.css');
    }

    public function process_payment($order_id)
    {
        $token = $this->get_payload_token_card($order_id);
        $host  = $this->host ? trim($this->host, ' \/') : self::CONTO_CARD_HOST;
        WC()->session->set('cq_order_id', $order_id);
        $data = [
            'token' => $token,
            'redirectUrlSuccess' => get_home_url() . '/wc-api/cq_card_success',
            'redirectUrlCancel' => get_home_url() . '/wc-api/cq_card_cancel',
            $data['webview'] = true,
        ];
        if ($this->embedded == 'embedded') {
            $data['token']              = $token;
            $data['host']               = $host;
            $data['redirectUrlSuccess'] = get_home_url() . '/wc-api/cq_card_success';
            $data['redirectUrlCancel']  = get_home_url() . '/wc-api/cq_card_cancel';
            $data['webview']            = true;
            wp_send_json($data);
        }

        if ($this->use_wc_blocks == 'yes') {
            $data['host'] = $host;
            wp_send_json($data);
        }

        $redirect = $host . '/initialize?' . http_build_query($data);

        return [
            'result' => 'success',
            'redirect' => $redirect,
        ];
    }

    /**
     * Conto Quickpay success Payment process.
     */
    public function webhook_cq_card_success()
    {
        global $woocommerce;
        $order_id = WC()->session->get('cq_order_id');
        $this->log('Success redirect page: session cq_order_id = ' . $order_id);
        $order_status = 'wc-pending';
        $order        = wc_get_order($order_id);

        if (!empty($order)) {
            $current_status = $order->get_status();
            if (
                $current_status !== 'wc-processing'
                && $current_status !== 'wc-completed'
                && $current_status !== 'processing'
                && $current_status !== 'completed'
            ) {
                $order->set_status($order_status);
            }

            $woocommerce->cart->empty_cart();
            $order->save();
            WC()->session->__unset('cq_order_id');

            if (isset(WC()->cart)) {
                WC()->cart->empty_cart();
            }

            $this->log('Order id: ' . $order_id . ' was successfully placed.');
            wp_redirect($this->get_return_url($order));
            exit();
        } else {
            wp_redirect(home_url());
            $this->log('Can\'t get an Order, make sure the transaction ID is correct.', 'critical');
        }
    }

    /**
     * Conto Quickpay cancel Payment process.
     */
    public function webhook_cq_card_cancel()
    {
        $order_id     = WC()->session->get('cq_order_id');
        $order_status = 'wc-cancelled';
        $order        = wc_get_order($order_id);
        if (!empty($order)) {
            $current_status = $order->get_status();
            if ($current_status != 'wc-processing' || $current_status != 'wc-completed') {
                $order->set_status($order_status);
            }
            $order->save();
            WC()->session->__unset('cq_order_id');
            $this->log('Order id: ' . $order_id . ' was canceled.');
            wp_redirect(wc_get_checkout_url());
            exit();
        } else {
            wp_redirect(home_url());
            $this->log('Can\'t get an Order, make sure the transaction ID is correct.', 'critical');
        }
    }

    /**
     * Conto Quickpay callback Payment process.
     */
    public function webhook_callback()
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $CallbackData = $this->getCallbackData();
            $transactions = $CallbackData->getTransactios();
            if (!empty($transactions)) {
                $wrong_transaction = [];
                foreach ($transactions as $transaction) {
                    if ($transaction->status !== CallbackData::STATUS_CONFIRMED && $transaction->status !== CallbackData::STATUS_SIGNED && $transaction->status !== CallbackData::STATUS_EXPIRED && $transaction->status !== CallbackData::STATUS_FAILED) {
                        array_push($wrong_transaction, $transaction->transactionId);
                        continue;
                    }
                    $order_id           = wc_get_order_id_by_order_key('wc_order_' . $transaction->transactionId);
                    $status_transaction = wc_strtolower($transaction->status);

                    switch ($status_transaction) {
                        case 'confirmed':
                            $tr_status = $this->get_option('confirmed');
                            break;
                        case 'signed':
                            $tr_status = $this->get_option('signed');
                            break;
                        case 'expired':
                            $tr_status = $this->get_option('expired');
                            break;
                        case 'failed':
                            $tr_status = $this->get_option('failed');
                            break;
                        default:
                            array_push($wrong_transaction, $transaction->transactionId);
                            break;
                    }

                    $order = wc_get_order($order_id);
                    if (!empty($order)) {
                        $callback_status = str_replace([ 'wc-', ' ' ], [
                            '',
                            '-',
                        ], $tr_status);

                        $current_status = $order->get_status();

                        if ($current_status === $callback_status) {
                            array_push($wrong_transaction, $transaction->transactionId);
                            $this->log('Can\'t change the status because it\'s the same as the current one. Transaction [' . $transaction->transactionId . ']', 'critical');
                            continue;
                        }

                        if (
                            $current_status == str_replace([ 'wc-', ' ' ], [ '', '-' ], $this->get_option('confirmed'))
                            && $callback_status == str_replace([ 'wc-', ' ' ], [ '', '-' ], $this->get_option('failed'))
                        ) {
                            continue;
                        }

                        $order->set_status($callback_status);
                        $order->save();
                        $this->log('Callback Payment process: Order id = ' . $order_id . '. Order key = ' . $transaction->transactionId . ' Transaction status = ' . $status_transaction);
                    } else {
                        array_push($wrong_transaction, $transaction->transactionId);
                        $this->log('Wrong transactions[' . implode(', ', $wrong_transaction) . ']', 'critical');
                    }
                }
                if (!empty($wrong_transaction)) {
                    $this->sendFailResponse('Wrong transactions[' . implode(', ', $wrong_transaction) . ']');
                } else {
                    $this->sendOkResponse();
                }
            } else {
                $this->sendFailResponse('Transactions are empty.');
            }
        } else {
            $this->log('Request method \'' . $_SERVER['REQUEST_METHOD'] . '\' not allowed. Only POST can be used.', 'critical');
            $this->sendFailResponse('Request method \'' . $_SERVER['REQUEST_METHOD'] . '\' not allowed. Only POST can be used.', 404);
            wp_redirect(home_url());
            exit();
        }
    }
}
