<?php

class ControllerExtensionPaymentQuickpay extends Controller
{
    const EMPTY_SIGN = '';
    const SSL = true;

    const OPENCART_ORDER_ID = 'order_id';
    const OPENCART_ORDER_CURRENCY_CODE = 'currency_code';
    const OPENCART_ORDER_TOTAL_CART = 'total';
    const OPENCART_ORDER_INVOICE_PREFIX = 'invoice_prefix';
    const OPENCART_ORDER_INVOICE_NO = 'invoice_no';
    const OPENCART_ORDER_COUNTRY_CODE = 'payment_iso_code_2';
    const OPENCART_ORDER_EMAIL = 'email';

    const OPENCART_CHECKOUT_ORDER_MODEL = 'checkout/order';

    const QUICKPAY_ACCEPTABLE_CURRENCY = 'EUR';

    const QUICKPAY_JWT_SECRET = 'payment_quickpay_jwt_secret';
    const QUICKPAY_JWT_ALGORITHM = 'payment_quickpay_jwt_signature_algorithm';
    const QUICKPAY_CONTRACT_ID = 'payment_quickpay_contract_id';
    const QUICKPAY_WIDGET_HOST_URL = 'payment_quickpay_widget_host_url';
    const QUICKPAY_SHOP_NAME = 'payment_quickpay_shop_name';
    const QUICKPAY_SHOP_URL = 'payment_quickpay_shop_url';
    const QUICKPAY_SEND_ON_SUCCESS = 'payment_quickpay_send_on_success';
    const QUICKPAY_SEND_ON_FAILURE = 'payment_quickpay_send_on_failure';

    const QUICKPAY_EXTENSION_ROUTE = 'extension/payment/quickpay';
    const QUICKPAY_PAYMENT_METHOD_CONFIRM_ROUTE = 'extension/payment/quickpay/confirm';
    const OPENCART_REDIRECT_SUCCESS_ROUTE = 'checkout/success';
    const OPENCART_REDIRECT_CANCEL_ROUTE = 'checkout/cart';
    const OPENCART_ERROR_NOT_FOUND_ROUTE = 'error/not_found';

    const QUICKPAY_REDIRECT_SUCCESS_PATH = 'index.php?route=extension/payment/quickpay/success';
    const QUICKPAY_REDIRECT_CANCEL_PATH = 'index.php?route=extension/payment/quickpay/cancel';

    const QUICKPAY_PENDING_ORDER_STATUS = 'payment_quickpay_pending_order_status';
    const QUICKPAY_PROCESSING_ORDER_STATUS = 'payment_quickpay_processing_order_status';
    const QUICKPAY_SIGNED_ORDER_STATUS = 'payment_quickpay_signed_order_status';
    const QUICKPAY_CONFIRMED_ORDER_STATUS = 'payment_quickpay_confirmed_order_status';
    const QUICKPAY_CANCELED_ORDER_STATUS = 'payment_quickpay_canceled_order_status';
    const QUICKPAY_EXPIRED_ORDER_STATUS = 'payment_quickpay_expired_order_status';
    const QUICKPAY_ACTIVE_ORDER_STATUS = 'payment_quickpay_active_order_status';
    const QUICKPAY_UNRESOLVED_ORDER_STATUS = 'payment_quickpay_unresolved_order_status';
    const QUICKPAY_SINGLE_BANK_SELECTION = 'payment_quickpay_single_bank';
    const PAYMENT_QUICKPAY_OTHER_BANKS_SELECTION = 'payment_quickpay_other_banks_selection';
    const QUICKPAY_REMITTANCE_INFORMATION = 'payment_quickpay_remittance_information';
    const QUICKPAY_PERIODIC_PAYMENT_STATUS = 'payment_quickpay_periodic_payment_status';

    const QUICKPAY_CALLBACK_RESPONSE_STATUS_FAILURE_CODE = 400;
    const QUICKPAY_CALLBACK_RESPONSE_STATUS_CONTENT_TYPE = 'Content-Type: application/json';
    const QUICKPAY_CALLBACK_REQUEST_BODY_TOKEN_FIELD = 'token';
    const QUICKPAY_CALLBACK_REQUEST_BODY_PATH = 'php://input';

    const QUICKPAY_VIEW_TEMPLATE = '/template/extension/payment/quickpay';
    const OPENCART_CONFIG_TEMPLATE = 'config_template';
    const QUICKPAY_PERIODIC_PAYMENT_START_DATE = 'quickpay_periodic_start_date';
    const QUICKPAY_PERIODIC_PAYMENT_END_DATE = 'quickpay_periodic_end_date';
    const QUICKPAY_PERIODIC_PAYMENT_FREQUENCY = 'quickpay_periodic_frequency';

    public function __construct($registry)
    {
        parent::__construct($registry);

        if (version_compare(phpversion(), '7.1', '>=')) {
            ini_set('serialize_precision', -1);
        }
    }

    public function index()
    {
        $this->load->language(self::QUICKPAY_EXTENSION_ROUTE);
        $this->load->model(self::OPENCART_CHECKOUT_ORDER_MODEL);
        $orderID = $this->session->data[self::OPENCART_ORDER_ID];
        $order = $this->model_checkout_order->getOrder($orderID);

        if ($order[self::OPENCART_ORDER_CURRENCY_CODE] == self::QUICKPAY_ACCEPTABLE_CURRENCY) {
            $data['action'] = $this->url->link(self::QUICKPAY_PAYMENT_METHOD_CONFIRM_ROUTE, self::EMPTY_SIGN, self::SSL);
            $data['text_quickpay_selection'] = $this->language->get('text_quickpay_selection');
        } else {
            $data['text_error_quickpay_selection'] = $this->language->get('text_quickpay_acceptable_currency') . self::QUICKPAY_ACCEPTABLE_CURRENCY;
        }

        if (file_exists(DIR_TEMPLATE . $this->config->get(self::OPENCART_CONFIG_TEMPLATE) . self::QUICKPAY_VIEW_TEMPLATE)) {
            return $this->load->view($this->config->get(self::OPENCART_CONFIG_TEMPLATE) . self::QUICKPAY_VIEW_TEMPLATE, $data);
        } else {
            return $this->load->view(self::QUICKPAY_EXTENSION_ROUTE, $data);
        }
    }

    public function success()
    {
        $this->redirect(self::OPENCART_REDIRECT_SUCCESS_ROUTE);
    }

    public function cancel()
    {
        $this->redirect(self::OPENCART_REDIRECT_CANCEL_ROUTE);
    }

    public function redirect($route)
    {
        $orderID = $this->request->get[self::OPENCART_ORDER_ID];

        $this->load->model(self::OPENCART_CHECKOUT_ORDER_MODEL);
        $order = $this->model_checkout_order->getOrder($orderID);

        if ($this->isPending($order['order_status_id'])) {
            $orderStatus = $this->config->get(self::QUICKPAY_PROCESSING_ORDER_STATUS);
            $this->model_checkout_order->addOrderHistory($orderID, $orderStatus);
        }
        $this->response->redirect($this->url->link($route, self::EMPTY_SIGN, self::SSL));
    }

    public function confirm()
    {
        $this->includeQuickpayLib();
        $this->load->model(self::OPENCART_CHECKOUT_ORDER_MODEL);
        $orderID = $this->session->data[self::OPENCART_ORDER_ID];
        $order = $this->model_checkout_order->getOrder($orderID);

        if ($order[self::OPENCART_ORDER_CURRENCY_CODE] == self::QUICKPAY_ACCEPTABLE_CURRENCY) {
            $order[self::QUICKPAY_SINGLE_BANK_SELECTION] = $_POST[self::QUICKPAY_SINGLE_BANK_SELECTION] ?? null;
            $order[self::QUICKPAY_PERIODIC_PAYMENT_STATUS] = $_POST[self::QUICKPAY_PERIODIC_PAYMENT_STATUS] ?? null;
            $order[self::QUICKPAY_PERIODIC_PAYMENT_START_DATE] = $_POST[self::QUICKPAY_PERIODIC_PAYMENT_START_DATE] ?? null;
            $order[self::QUICKPAY_PERIODIC_PAYMENT_END_DATE] = $_POST[self::QUICKPAY_PERIODIC_PAYMENT_END_DATE] ?? null;
            $order[self::QUICKPAY_PERIODIC_PAYMENT_FREQUENCY] = $_POST[self::QUICKPAY_PERIODIC_PAYMENT_FREQUENCY] ?? null;

            $orderStatus = $this->config->get(self::QUICKPAY_PENDING_ORDER_STATUS);
            $this->model_checkout_order->addOrderHistory($orderID, $orderStatus);

            $this->loadWidget($order);
        } else {
            exit($this->language->get('quickpay_acceptable_currency') . self::QUICKPAY_ACCEPTABLE_CURRENCY);
        }
    }

    public function callback()
    {
        if ($this->request->server['REQUEST_METHOD'] == 'POST') {
            $this->includeQuickpayJWT();
            $data = $this->getCallbackData();
            $orderID = $data->getTransactionId();
            $this->load->model(self::OPENCART_CHECKOUT_ORDER_MODEL);
            $order = $this->model_checkout_order->getOrder($orderID);
            if (empty($order)) {
                $this->sendFailResponse('Provided \'transactionId\' Does Not Exist');
                exit();
            }

            $this->changeOrderStatusFromCallback($orderID, $order['order_status_id'], $data->getStatus());
            $this->sendOkResponse();
            exit();
        } else {
            $this->response->redirect($this->url->link(self::OPENCART_ERROR_NOT_FOUND_ROUTE, self::EMPTY_SIGN, self::SSL));
        }
    }

    private function changeOrderStatusFromCallback($orderID, $currentStatusId, $callbackStatus)
    {
        $newStatus = null;
        switch ($callbackStatus) {
            case CallbackData::STATUS_SIGNED:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId) || $this->isCancelled($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_SIGNED_ORDER_STATUS);
                }
                break;
            case CallbackData::STATUS_CONFIRMED:
                if ($this->isProcessing($currentStatusId) || $this->isSigned($currentStatusId) || $this->isPending($currentStatusId)) {
                    if ($this->config->get(self::QUICKPAY_CONFIRMED_ORDER_STATUS) != $this->config->get(self::QUICKPAY_SIGNED_ORDER_STATUS)) {
                        $newStatus = $this->config->get(self::QUICKPAY_CONFIRMED_ORDER_STATUS);
                    }
                }
                break;
            case CallbackData::STATUS_FAILED:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_CANCELED_ORDER_STATUS);
                }
                break;
            case CallbackData::STATUS_EXPIRED:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_EXPIRED_ORDER_STATUS);
                }
                break;
            case CallbackData::STATUS_ACTIVE:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId) || $this->isCancelled($currentStatusId)
                    || $this->isUnresolved($currentStatusId) || $this->isSigned($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_ACTIVE_ORDER_STATUS);
                }
                break;
            case CallbackData::STATUS_UNRESOLVED:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId) || $this->isCancelled($currentStatusId)
                    || $this->isSigned($currentStatusId) || $this->isActive($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_UNRESOLVED_ORDER_STATUS);
                }
                break;
            case CallbackData::STATUS_CANCELLED:
                if ($this->isProcessing($currentStatusId) || $this->isPending($currentStatusId) || $this->isSigned($currentStatusId)
                    || $this->isActive($currentStatusId) || $this->isUnresolved($currentStatusId)) {
                    $newStatus = $this->config->get(self::QUICKPAY_CANCELED_ORDER_STATUS);
                }
                break;
        }
        if (isset($newStatus)) {
            $this->model_checkout_order->addOrderHistory($orderID, $newStatus);
        }
    }

    private function getCallbackData()
    {
        $jwtSecret = $this->getJwtSecret();
        $jwtSignAlg = $this->getJwtSignAlgorithm();
        try {
            $jwtToken = $this->parseRequestBody();
            return new CallbackData(JWTUtil::decode($jwtToken, $jwtSecret, $jwtSignAlg));
        } catch (JWTDecodeException $e) {
            $this->sendFailResponse('JWT Decode Error: \'' . $e->getMessage() . '\'');
        } catch (CallbackParseException $e) {
            $this->sendFailResponse('Callback Data Parse Error: \'' . $e->getMessage() . '\'');
        }
        exit();
    }

    private function parseRequestBody()
    {
        try {
            set_error_handler(function ($severity, $message) {
                throw new Exception($message, $severity);
            });
            $body = file_get_contents(self::QUICKPAY_CALLBACK_REQUEST_BODY_PATH);
            $json = json_decode($body, true);
            $jwtToken = $json[self::QUICKPAY_CALLBACK_REQUEST_BODY_TOKEN_FIELD];
            restore_error_handler();
            return $jwtToken;
        } catch (Exception $e) {
            $this->sendFailResponse('POST Request Body Parse Error: \'' . $e->getMessage() . '\'');
            exit();
        }
    }

    private function sendOkResponse()
    {
        header_remove();
        header(self::QUICKPAY_CALLBACK_RESPONSE_STATUS_CONTENT_TYPE);
        echo json_encode(['status' => 'ok']);
    }

    private function sendFailResponse($message)
    {
        header_remove();
        header(self::QUICKPAY_CALLBACK_RESPONSE_STATUS_CONTENT_TYPE);
        http_response_code(self::QUICKPAY_CALLBACK_RESPONSE_STATUS_FAILURE_CODE);
        echo json_encode(['status' => 'failure', 'message' => $message]);
    }

    private function loadWidget(array $order)
    {
        $secret = $this->getJwtSecret();
        $signAlg = $this->getJwtSignAlgorithm();
        $jwtPayload = $this->getJwtPayload($order);
        $widgetConfig = $this->getWidgetConfig($order);
        $widgetHostUrl = $this->getWidgetHostUrl();
        WidgetLoader::factory()
            ->setWidgetConfig($widgetConfig)
            ->setToken(JWTUtil::encode($jwtPayload, $secret, $signAlg))
            ->setWidgetHostUrl($widgetHostUrl)
            ->load();
    }

    private function getJwtPayload($order)
    {
        $enablePeriodic = (true == $order[self::QUICKPAY_PERIODIC_PAYMENT_STATUS]);

        $payload = JWTPayload::factory()
            ->setAmount($order[self::OPENCART_ORDER_TOTAL_CART])
            ->setContractId($this->getQuickpayContractId())
            ->setPaymentPurpose($this->getPaymentPurpose($order, $enablePeriodic))
            ->setTransactionId($order[self::OPENCART_ORDER_ID])
            ->setCheckoutId($order[self::OPENCART_ORDER_ID])
            ->setSendOnSuccess($this->getSendOnSuccess())
            ->setSendOnFailure($this->getSendOnFailure())
            ->setShopName($this->getShopName())
            ->setShopUrl($this->getShopUrl())
            ->setAddress($this->encrypt($order[self::OPENCART_ORDER_EMAIL]));

        if ($enablePeriodic) {
            $payload->setEnablePeriodicPayment(true)
                ->setServiceType('PERIODIC')
                ->setPeriodicStartDate($order[self::QUICKPAY_PERIODIC_PAYMENT_START_DATE])
                ->setPeriodicFrequency($order[self::QUICKPAY_PERIODIC_PAYMENT_FREQUENCY]);

            if ($order[self::QUICKPAY_PERIODIC_PAYMENT_END_DATE]) {
                $payload->setPeriodicEndDate($order[self::QUICKPAY_PERIODIC_PAYMENT_END_DATE]);
            }
        }

        return $payload->create();
    }


    private function getPaymentPurpose($order, $enablePeriodic)
    {
        $paymentPurposeBasic = $order[self::OPENCART_ORDER_INVOICE_PREFIX] . '-' . $order[self::OPENCART_ORDER_INVOICE_NO] . "-" . $order[self::OPENCART_ORDER_ID];

        if ($enablePeriodic) {
            $paymentPurpose = $this->config->get(self::QUICKPAY_REMITTANCE_INFORMATION) ?: $paymentPurposeBasic;
        } else {
            $paymentPurpose = $paymentPurposeBasic;
        }

        return $paymentPurpose;
    }

    private function getWidgetConfig($order)
    {
        $creditorValue = $_POST[self::QUICKPAY_SINGLE_BANK_SELECTION];
        if (($creditorValue !== "null") && !$this->getOtherBanks()) {
            $usesKevin = 'false';
        } elseif (($creditorValue !== "null") && $this->getOtherBanks()) {
            $usesKevin = 'null';
        } else {
            $usesKevin = $this->getOtherBanks() ? 'true' : 'false';
        }

        $orderParam = '&' . self::OPENCART_ORDER_ID . '=' . $order[self::OPENCART_ORDER_ID];
        return WidgetConfig::factory()
            ->setRedirectUrlSuccess(HTTPS_SERVER . $this::QUICKPAY_REDIRECT_SUCCESS_PATH . $orderParam)
            ->setRedirectUrlCancel(HTTPS_SERVER . $this::QUICKPAY_REDIRECT_CANCEL_PATH . $orderParam)
            ->setDefaultLanguage($this->getLanguageCode($this->language->get('code')))
            ->setDefaultCountry($order[self::OPENCART_ORDER_COUNTRY_CODE])
            ->setCreditor($order[self::QUICKPAY_SINGLE_BANK_SELECTION])
            ->setUsesKevin($usesKevin)
            ->create();
    }

    private function getQuickpayContractId()
    {
        return $this->config->get(self::QUICKPAY_CONTRACT_ID);
    }

    private function getJwtSecret()
    {
        return $this->config->get(self::QUICKPAY_JWT_SECRET);
    }

    private function getOtherBanks()
    {
        return (bool)$this->config->get(self::PAYMENT_QUICKPAY_OTHER_BANKS_SELECTION);
    }

    private function getJwtSignAlgorithm()
    {
        return $this->config->get(self::QUICKPAY_JWT_ALGORITHM);
    }

    private function getWidgetHostUrl()
    {
        return $this->config->get(self::QUICKPAY_WIDGET_HOST_URL);
    }

    private function getShopName()
    {
        return $this->config->get(self::QUICKPAY_SHOP_NAME);
    }

    private function getShopUrl()
    {
        return $this->config->get(self::QUICKPAY_SHOP_URL);
    }

    private function getSendOnSuccess()
    {
        return (boolean)json_decode(strtolower($this->config->get(self::QUICKPAY_SEND_ON_SUCCESS)));
    }

    private function getSendOnFailure()
    {
        return (boolean)json_decode(strtolower($this->config->get(self::QUICKPAY_SEND_ON_FAILURE)));
    }

    private function getLanguageCode($lang)
    {
        switch ($lang) {
            case 'en':
                return 'ENG';
            case 'ru':
                return 'RUS';
            default:
                return 'LIT';
        }
    }

    private function includeQuickpayLib()
    {
        if (!class_exists('IncludeQuickpayLib')) {
            require_once __DIR__ . '/quickpay/include.php';
        }
    }

    private function includeQuickpayJWT()
    {
        if (!class_exists('IncludeQuickpayJWT')) {
            require_once __DIR__ . '/quickpay/jwt/include.php';
        }
    }

    /**
     * @param $currentStatusId
     *
     * @return bool
     */
    private function isProcessing($currentStatusId)
    {
        return $currentStatusId == $this->config->get(self::QUICKPAY_PROCESSING_ORDER_STATUS);
    }

    /**
     * @param $currentStatus
     *
     * @return bool
     */
    private function isSigned($currentStatus)
    {
        return $currentStatus == $this->config->get(self::QUICKPAY_SIGNED_ORDER_STATUS);
    }

    /**
     * @param $currentStatus
     *
     * @return bool
     */
    private function isPending($currentStatus)
    {
        return $currentStatus == $this->config->get(self::QUICKPAY_PENDING_ORDER_STATUS);
    }

    private function isCancelled($currentStatus)
    {
        return $currentStatus == $this->config->get(self::QUICKPAY_CANCELED_ORDER_STATUS);
    }

    private function isActive($currentStatus)
    {
        return $currentStatus == $this->config->get(self::QUICKPAY_ACTIVE_ORDER_STATUS);
    }

    private function isUnresolved($currentStatus)
    {
        return $currentStatus == $this->config->get(self::QUICKPAY_UNRESOLVED_ORDER_STATUS);
    }

    private function encrypt($data)
    {
        $cipherAlgorithm = 'aes-256-ctr';

        $secretKey = $this->getJwtSecret();
        $passphrase = openssl_digest($secretKey, 'sha256');

        $ivSize = openssl_cipher_iv_length($cipherAlgorithm);
        $iv = openssl_random_pseudo_bytes($ivSize);

        $cipherOptions = OPENSSL_RAW_DATA;

        $cipherText = openssl_encrypt(
            $data,
            $cipherAlgorithm,
            $passphrase,
            $cipherOptions,
            $iv
        );

        $b64EncodedCipherText = base64_encode($iv . $cipherText);

        return $b64EncodedCipherText;
    }
}
